#region References

using System;
using System.Collections;
using System.Data;
using System.Data.SqlClient;
using System.Text;

using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;

using gov.va.med.vbecs.Common;
using gov.va.med.vbecs.DAL;
using gov.va.med.vbecs.DAL.VistALink.Client;
using gov.va.med.vbecs.DAL.VistALink.OpenLibrary;

#endregion

namespace gov.va.med.vbecs.DAL.VAL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Brian    lin</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>11/30/2006</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>
	///Performs an exchange of post transfusion data with the Blood Bank DSS Extract software in VistA via VistALink.
	///</summary>

	#endregion

	public class DssExtract
	{
		#region Variables 
		
		private const string DSS_EXTRACT_RPC_NAME = "VBECS DSS EXTRACT";
		
		private const string COMPONENT_ABB_PARAM_NAME = "COMPONENT ABBREVIATION";
		private const string DFN_PARAM_NAME = "DFN";
		private const string NUMBER_OF_UNITS_PARAM_NAME = "NUMBER OF UNITS";
		private const string ORDERING_LOCATION_PARAM_NAME = "ORDERING LOCATION";
		private const string ORDERING_PROVIDER_PARAM_NAME = "ORDERING PROVIDER";
		private const string PHYSICIAN_PARAM_NAME = "PHYSICIAN";
		private const string PRODUCT_NAME_PARAM_NAME = "PRODUCT NAME";
		private const string REACTION_INDICATOR_PARAM_NAME = "REACTION";
		private const string REACTION_TYPE_PARAM_NAME = "REACTION TYPE";
		private const string TRANSACTION_ID_PARAM_NAME = "TRANSACTION ID";
		private const string TRANSFUSION_DATE_PARAM_NAME = "TRANSFUSION DATE";
		private const string TRANSFUSION_LOCATION_PARAM_NAME = "TRANSFUSION LOCATION";
		private const string UNIT_MODIFICATION_PARAM_NAME = "UNIT MODIFICATION";
		private const string VOLUME_PARAM_NAME = "VOLUME";

		private const string SUCCESS_INDICATOR_TRUE = "1";

		#endregion

		#region Public Methods
		
		///<Developers>
		///	<Developer>Brian    lin, David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>12/5/2006</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8460"> 
		///		<ExpectedInput>Valid PatientTransfusion Guid</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8461"> 
		///		<ExpectedInput>Invalid PatientTransfusion Guid</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// UpdateVistADssExtract
		/// </summary>
		/// <param name="patientTransfusionGuid">Patient change unique identifier; optional (use Guid.Empty)</param>
		/// <param name="lastUpdateFunctionId">Last update function id</param>
		/// <returns></returns>
		public static bool UpdateVistADssExtract( Guid patientTransfusionGuid, UpdateFunction lastUpdateFunctionId )
		{
			bool returnValue = true;
			//
            try
            {
                DataTable dtDssExtract = GetDssExtractDataTable(patientTransfusionGuid);
                //
                if (dtDssExtract != null && dtDssExtract.Rows != null && dtDssExtract.Rows.Count > 0)
                {
                    for (int idx = 0; idx < dtDssExtract.Rows.Count; idx++)
                    {
                        #region Parameter Table

                        Hashtable parameterTable = new Hashtable();
                        parameterTable.Add(TRANSACTION_ID_PARAM_NAME, dtDssExtract.Rows[idx][TABLE.PatientTransfusion.PatientTransfusionGuid].ToString().Trim());
                        parameterTable.Add(DSS_EXTRACT_RPC_NAME, DSS_EXTRACT_RPC_NAME);
                        parameterTable.Add(DFN_PARAM_NAME, dtDssExtract.Rows[idx][TABLE.Patient.VistaPatientId].ToString().Trim());
                        parameterTable.Add(ORDERING_LOCATION_PARAM_NAME, dtDssExtract.Rows[idx][ARTIFICIAL.OrderingLocation].ToString().Trim());
                        parameterTable.Add(TRANSFUSION_LOCATION_PARAM_NAME, dtDssExtract.Rows[idx][TABLE.PatientTransfusion.DivisionCode].ToString().Trim());
                        parameterTable.Add(PHYSICIAN_PARAM_NAME, dtDssExtract.Rows[idx][TABLE.IssuedUnit.PhysicianName].ToString().Trim());
                        parameterTable.Add(ORDERING_PROVIDER_PARAM_NAME, dtDssExtract.Rows[idx][TABLE.PatientOrder.OrderingProviderId].ToString().Trim());
                        parameterTable.Add(PRODUCT_NAME_PARAM_NAME, dtDssExtract.Rows[idx][TABLE.BloodProduct.ProductShortName].ToString().Trim());
                        parameterTable.Add(COMPONENT_ABB_PARAM_NAME, dtDssExtract.Rows[idx][TABLE.ComponentClass.ComponentClassShortName].ToString().Trim());
                        parameterTable.Add(NUMBER_OF_UNITS_PARAM_NAME, dtDssExtract.Rows[idx][ARTIFICIAL.TransfusionPooledUnitsCount].ToString().Trim());
                        parameterTable.Add(TRANSFUSION_DATE_PARAM_NAME, dtDssExtract.Rows[idx][TABLE.PatientTransfusion.TransfusionEndDateTime].ToString().Trim());
                        parameterTable.Add(VOLUME_PARAM_NAME, dtDssExtract.Rows[idx][TABLE.PatientTransfusion.TransfusedVolume].ToString().Trim());
                        parameterTable.Add(REACTION_TYPE_PARAM_NAME, dtDssExtract.Rows[idx][TABLE.TransfusionReactionType.TransfusionReactionTypeText].ToString().Trim());
                        parameterTable.Add(UNIT_MODIFICATION_PARAM_NAME, dtDssExtract.Rows[idx][ARTIFICIAL.UnitModifications].ToString().Trim());
                        parameterTable.Add(REACTION_INDICATOR_PARAM_NAME, dtDssExtract.Rows[idx][TABLE.PatientTransfusion.ReactionIndicator].ToString().Trim());

                        #endregion
                        //
                        RpcRequest dssExtractRequest = VistALink.CreateRpcRequest(DSS_EXTRACT_RPC_NAME);
                        //
                        RpcArrayParameter rpcArray = new RpcArrayParameter(1, parameterTable);
                        //
                        dssExtractRequest.Parameters.Add(rpcArray);
                        //
                        string rpcResult = VistALink.GetRpcBroker().ExecuteRpc(dssExtractRequest);
                        //
                        DataTable currentTable = DataTransformUtility.LoadXmlStringIntoSingleDataTable(rpcResult);
                        //
                        #region Check Results

                        bool successIndicator = true;

                        for (int idxResults = 0; idxResults < currentTable.Rows.Count; idxResults++)
                        {
                            if (currentTable.Rows[idxResults].Table.Columns.Contains(ARTIFICIAL.SuccessIndicator)
                                && !currentTable.Rows[idxResults].IsNull(ARTIFICIAL.SuccessIndicator))
                            {
                                if (currentTable.Rows[idxResults][ARTIFICIAL.SuccessIndicator].ToString() != SUCCESS_INDICATOR_TRUE)
                                {
                                    returnValue = false;
                                    successIndicator = false;
                                    break;
                                }
                            }
                        }
                        //
                        UpdateDssExtractSuccessIndicator((Guid)dtDssExtract.Rows[idx][TABLE.PatientTransfusion.PatientTransfusionGuid], (byte[])dtDssExtract.Rows[idx][TABLE.PatientTransfusion.RowVersion], successIndicator, lastUpdateFunctionId);

                        #endregion
                    }
                }
            }
            catch (Common.DataAccessLayerException)
            {
                // Suppress row-version exception (extremely unlikely)
            }
			//
			return returnValue;
		}

		///<Developers>
		///	<Developer>David Askew</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/11/2007</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8466"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Guids</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8467"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retreives patient transfusion guids that have not been processed successfully 
		/// </summary>
		/// <returns>Guid []</returns>
		public static Guid [] GetPatientTransfusionGuidsForDssExtract()
		{
			Guid [] guids = null;
			//
			SqlParameter[] prms =
			{
				new SqlParameter( STOREDPROC.GetPatientTransfusionGuidsForDssExtract.divisioncode, System.Data.SqlDbType.Char )
			};
			//
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			//
			DataSet dsGuids = Common.StoredProcedure.GetData( STOREDPROC.GetPatientTransfusionGuidsForDssExtract.StoredProcName, prms );
			//
			if ( dsGuids != null && dsGuids.Tables != null && dsGuids.Tables.Count == 1 &&  dsGuids.Tables[0].Rows != null && dsGuids.Tables[0].Rows.Count > 0 )
			{
				int rowCount = dsGuids.Tables[0].Rows.Count;
				//
				guids = new Guid[rowCount];
				//
				for ( int idx = 0; idx < rowCount; idx++ )
				{
					guids[idx] = (Guid)dsGuids.Tables[0].Rows[idx][TABLE.PatientTransfusion.PatientTransfusionGuid];
				}
			}
			//
			return guids;
		}

		#endregion

		#region Private Methods
		
		/// <summary>
		/// Retreives data for dss extract based on patient transfusion guid
		/// </summary>
		/// <param name="patientTransfusionGuid">Patient change unique identifier; optional (use Guid.Empty)</param>
		/// <returns>DataTable</returns>
		private static DataTable GetDssExtractDataTable( Guid patientTransfusionGuid )
		{
			SqlParameter [] prms = null;
			//
			if ( patientTransfusionGuid == Guid.Empty )
			{
				prms = new SqlParameter[1];
				//
				prms[0] = new SqlParameter( STOREDPROC.GetDssExtract.divisioncode, System.Data.SqlDbType.Char );
				//
				prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			}
			else
			{
				prms = new SqlParameter[2];
				//
				prms[0] = new SqlParameter( STOREDPROC.GetDssExtract.divisioncode, System.Data.SqlDbType.Char );
				prms[1] = new SqlParameter( STOREDPROC.GetDssExtract.patienttransfusionguid, System.Data.SqlDbType.UniqueIdentifier );
				//
				prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
				prms[1].Value = patientTransfusionGuid;
			}
			//
			DataSet ds = Common.StoredProcedure.GetData( STOREDPROC.GetDssExtract.StoredProcName, prms );
			//
			#region Modification Data
			//
			// Check for modification data
			//
			ds.Tables[0].Columns.Add( ARTIFICIAL.UnitModifications, typeof(string) );
			//
			if ( ds.Tables.Count == 2 && ds.Tables[1] != null && ds.Tables[1].Rows != null && ds.Tables[1].Rows.Count > 0 )
			{
				for ( int mainIdx = 0; mainIdx < ds.Tables[0].Rows.Count; mainIdx++ )
				{
					Guid currentPatientTransfusionGuid = (Guid)ds.Tables[0].Rows[mainIdx][TABLE.PatientTransfusion.PatientTransfusionGuid];
					//
					StringBuilder modifications = new StringBuilder( string.Empty );
					//		
					for ( int modIdx = 0; modIdx < ds.Tables[1].Rows.Count; modIdx++ )
					{
						if ( currentPatientTransfusionGuid == (Guid)ds.Tables[1].Rows[modIdx][TABLE.PatientTransfusion.PatientTransfusionGuid] )
						{
							string currentMod = ds.Tables[1].Rows[modIdx][TABLE.BloodUnitModification.ProductModificationCode].ToString();
							//			
							if( modifications.ToString().IndexOf(currentMod) == -1 )
							{
								modifications.Append( currentMod );
							}
						}
					}
					//
					ds.Tables[0].Rows[mainIdx][ARTIFICIAL.UnitModifications] = modifications.ToString();
				}
			}
			//
			#endregion
			//
			return ds.Tables[0];
		}

		/// <summary>
		/// Updates the DssExtractSuccessIndicator on the Patient Transfusion table; 1 = success, 0 = failure
		/// </summary>
		/// <param name="patientTransfusionGuid">Patient transfusion unique identifier</param>
		/// <param name="rowVersion">Patient transfusion row version</param> 
		/// <param name="successIndicator">Success indicator</param> 
		/// <param name="lastUpdateFunctionId">Last update function id</param> 
		/// <returns>Success indicator</returns>
		private static bool UpdateDssExtractSuccessIndicator( Guid patientTransfusionGuid, byte [] rowVersion, bool successIndicator, UpdateFunction lastUpdateFunctionId )
		{
			if ( patientTransfusionGuid == Guid.Empty )
			{
				throw new ArgumentException( Common.StrRes.SysErrMsg.Common.InvalidFormat("patient transfusion identifier").ResString );
			}
			//
			SqlParameter[] prms =
			{
				new SqlParameter( STOREDPROC.UpdateDssExtractSuccessIndicator.patienttransfusionguid, System.Data.SqlDbType.UniqueIdentifier ),
				new SqlParameter( STOREDPROC.UpdateDssExtractSuccessIndicator.successindicator, System.Data.SqlDbType.Bit ),
				new SqlParameter( STOREDPROC.UpdateDssExtractSuccessIndicator.lastupdatefunctionid, System.Data.SqlDbType.Int ),
				new SqlParameter( STOREDPROC.UpdateDssExtractSuccessIndicator.lastupdateuser, System.Data.SqlDbType.VarChar ),
				new SqlParameter( STOREDPROC.UpdateDssExtractSuccessIndicator.rowversion, System.Data.SqlDbType.Timestamp )
			};
			//
			prms[0].Value = patientTransfusionGuid;
			prms[1].Value = successIndicator;
			prms[2].Value = (int)lastUpdateFunctionId;
			prms[3].Value = Common.LogonUser.LogonUserName;
			prms[4].Value = rowVersion;
			//
			return (((new StoredProcedure()).TransactionalGetValue(STOREDPROC.UpdateDssExtractSuccessIndicator.StoredProcName, prms)) == 0);
		}

		#endregion
	}
}
